<?php
// school_admission/public/branch_print_letter.php

// --- DEBUGGING LINES START ---
// These lines will force PHP errors to display in the browser.
// REMOVE or comment them out once your application is stable and deployed to production.
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);
// --- DEBUGGING LINES END ---


if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

require_once __DIR__ . '/../config/db.php';
require_once __DIR__ . '/../includes/functions.php';
require_once __DIR__ . '/../includes/auth_check.php';

requireRole(['branch_admin']); // Only Branch Admin can access this page

$message = '';
if (isset($_SESSION['message'])) {
    $message = $_SESSION['message'];
    unset($_SESSION['message']);
}

$current_branch_id = $_SESSION['branch_id'];

// Fetch applicants from the current branch whose admission letters have been generated
$applicants_with_letters = [];
$sql = "SELECT
            a.applicant_id,
            a.first_name,
            a.last_name,
            a.application_no,
            a.application_date,
            a.status,
            c.course_name,
            al.letter_id,
            al.generated_at,
            al.printed_at
        FROM
            applicants a
        JOIN
            courses c ON a.course_id = c.course_id
        JOIN
            admission_letters al ON a.applicant_id = al.applicant_id
        WHERE
            a.branch_id = ? AND a.status = 'admission_letter_generated'
        ORDER BY
            a.created_at DESC";

$stmt = $conn->prepare($sql);
if ($stmt) {
    $stmt->bind_param("i", $current_branch_id);
    $stmt->execute();
    $result = $stmt->get_result();
    while ($row = $result->fetch_assoc()) {
        $applicants_with_letters[] = $row;
    }
    $stmt->close();
} else {
    $message = "<div class='message error'>Error fetching applicants with letters: " . $conn->error . "</div>";
}

include __DIR__ . '/../includes/header.php';
?>

<h2 class="mt-30">Print Admission Letters</h2>

<?php if (!empty($message)): ?>
    <?php echo $message; ?>
<?php endif; ?>

<?php if (empty($applicants_with_letters)): ?>
    <div class="message info">No admission letters have been generated for applicants in your branch yet.</div>
<?php else: ?>
    <table>
        <thead>
            <tr>
                <th>App No.</th>
                <th>Name</th>
                <th>Course</th>
                <th>Letter Generated On</th>
                <th>Printed On</th>
                <th>Actions</th>
            </tr>
        </thead>
        <tbody>
            <?php foreach ($applicants_with_letters as $applicant): ?>
                <tr>
                    <td data-label="App No."><?php echo htmlspecialchars($applicant['application_no']); ?></td>
                    <td data-label="Name"><?php echo htmlspecialchars($applicant['first_name'] . ' ' . $applicant['last_name']); ?></td>
                    <td data-label="Course"><?php echo htmlspecialchars($applicant['course_name']); ?></td>
                    <td data-label="Generated On"><?php echo date('d-M-Y H:i', strtotime($applicant['generated_at'])); ?></td>
                    <td data-label="Printed On">
                        <?php echo $applicant['printed_at'] ? date('d-M-Y H:i', strtotime($applicant['printed_at'])) : 'Not Yet Printed'; ?>
                    </td>
                    <td data-label="Actions" class="action-links">
                        <a href="view_admission_letter.php?letter_id=<?php echo $applicant['letter_id']; ?>" target="_blank" class="btn btn-primary btn-sm"><i class="fas fa-print"></i> View/Print Letter</a>
                    </td>
                </tr>
            <?php endforeach; ?>
        </tbody>
    </table>
<?php endif; ?>

<?php
include __DIR__ . '/../includes/footer.php';
$conn->close();
?>